function [xf,IDopt]=nor2xf(om,ga,pb,cp,w,ind)

%NOR2XF computation of the FRF associated to a normal mode model
%
%	Synopsis: [xf,IDopt] = nor2xf(om,ga,pb,cp,w)
%		  [xf,IDopt] = nor2xf(om,ga,pb,cp,w,ind)
%
%	OM the modal stiffness which can be replaced by a column vector of
%	   modal frequencies FREQ (in rad/s)
%	GA the modal damping matrix can be replaced by a column vector of modal
%	   damping ratio DAMP or a single damping ratio to be applied to all
%	   modes
%	PB and CP the normal mode input and output shape matrices
%	W  specifies the frequency points at which the FRF should be computed
%
%	WARNING: W and FREQ should specify frequencies in rad/sec
%
%	High frequency modes can be truncated by specifying the indices IND
%	of modes to be kept. If modal frequencies FREQ (in rad/s) are given a
%	a static correction is then introduced to account for the low frequency
%	effects of truncated modes.
%
%	WARNING: The result xf is a displacement FRF. To obtain a velocity
%	output, specify displacement CPD and velocity CPV modal outputs and
%	use NOR2SS(OM,GA,PB,[CPD CPV]). You can also differentiate xf by
%	  xfv = j*xf.*w(:,ones(1,size(xf,2)))
%
%	See also NOR2SS, NOR2RES


%	Etienne Balmes  4/30/92, 11/30/94
%       Copyright (c) 1990-1994 by Etienne Balmes
%       All Rights Reserved.

[np,i1]=size(om);
if     i1==1 freq=om;   om=diag(om.^2);
elseif i1~=np error('OM is not square');
else freq=[];end

[i2,i3]=size(ga); damp=[];
if ~isempty(freq)
   if i2==1 & i3==1
	if np==1 disp('WARNING: damping ratio input assumed'); end
	damp=ga*ones(np,1);  ga=diag(2*freq.*damp);
   elseif i3==1
	damp=ga;             ga=diag(2*freq.*damp);
   end	
else
   if i1~=np|i2~=np error('GA is not consistent with OM'); end
end


if nargin<6 		ind=[1:np];		end % keeps all modes
if isempty(ind) 	ind=[1:np];		end % keeps all modes
ind=sort(ind);


[i4,na]=size(pb); if i4~=np error('PBS is not consistent with OM'); end
[ns,i7]=size(cp);

if i7==2*i4 	cpi=cp(:,1:(i7/2));
		cvi=cp(:,(i7/2+1):i7); % disp. and vel. sensors
elseif i7==i4 	cpi=cp(:,1:i7);cvi=zeros(ns,i4);
else 		error('The dimensions of cp and pb are incompatible');
end


w=w(:);  nw=length(w);

[NMode]=length(ind);

% static correction computation

if NMode<np
   if ~isempty(freq)
	disp('Static Correction');

	cind=[1:np-NMode]; l=1;
	for j1=1:np;  ik=0;
	    for j2=1:NMode;  if ind(j2)==j1 ik=1;  end;  end
	    if ik==0     cind(l)=j1;  l=l+1; end
	end

	finv=diag(freq(cind).^(-2));
	dd=cpi(:,cind)*finv*pb(cind,:);	dv=cvi(:,cind)*finv*pb(cind,:);
	dd=dd(:).';				dv=dv(:).';

   else
	disp('Static correction only available with modal frequencies FREQ');
	dd=zeros(1,na*ns);   dv=zeros(1,na*ns);
   end
else
  dd=zeros(1,na*ns);   dv=zeros(1,na*ns);
end;

% ----------------------------------------------------------------------------
% Case with proportional damping
% ----------------------------------------------------------------------------

if ~isempty(damp)==1
  
  i2 = find(w==0); if ~isempty(i2) & any(freq==0)
     i1=find(w);nw=length(i1); w = w(i1);
  end
  w1=2*j*w;   w2=-w.^2;  xf=zeros(nw,ns*na);
  for j1= 1:NMode

    tm=ones(nw,1)./(w2+w1*(freq(j1)*damp(j1))+freq(j1)^2);

    cfd=(cpi(:,ind(j1)))*pb(ind(j1),:);   cfd=cfd(:);
    xf=xf+tm*cfd';

    if any(any(cvi~=0))
	cfv=(cvi(:,ind(j1)))*pb(ind(j1),:);   cfv=cfv(:);
	xf=xf+tm.*w*j*cfv';
    end
  end
  if ~isempty(i2) & any(freq==0)
     w(i1)=w;w(i2)=zeros(length(i2),1);
     xf(i1,:)=xf;xf(i2,:) = 0*ones(length(i2),size(xf,2));
     disp('Warning: infinite response at w=0 set to zero');
  end

% ----------------------------------------------------------------------------
% Case with non-proportional damping
% ----------------------------------------------------------------------------

elseif isempty(damp)

%    disp('Non-proportional damping');
    xf=qbode([zeros(NMode) eye(NMode); -om(ind,ind) -ga(ind,ind)],...
	[zeros(NMode,na);pb(ind,:)],[cpi(:,ind) cvi(:,ind)],zeros(ns,na),w);

end

% ----------------------------------------------------------------------------

if any(any(dd~=0))
   xf=xf+ones(length(w),1)*dd;
end

if any(any(dv~=0))
   xf=xf+j*w*dv;
end


IDopt = [3 0 22 0 0 1 ns na 0 0 0  0];
