function [a,b,c,d]=nor2ss(om,ga,phib,cphi,ind,fc)

%NOR2SS	transformation from normal mode form NOR to the state-space form
%	with ability to truncate modes and introduce static correction modes
%	to account for the low frequency components of the truncated modes 
%
%	Synopsis: [a,b,c,d]=nor2ss(om,ga,phib,cphi)
%		  [a,b,c,d]=nor2ss(om,ga,phib,cphi,ind,fc)
%
%	NOR2SS create the state space model associated to the normal mode
%	model composed of
%
%	OM the modal stiffness which can be replaced by a column vector of
%	   modal frequencies FREQ (in rad/s)
%	GA the modal damping matrix can be replaced by a column vector of modal
%	   damping ratio DAMP or a single damping ratio to be applied to all
%	   modes
%	PHIB and CPHI the normal mode input and output shape matrices
%
%	High frequency modes can be truncated by specifying the indices IND.
%	nor2ss(om,ga,phib,cphi,ind) is the same as nor2ss(om(ind,ind),
%	ga(ind,ind),phib(ind,:),cphi(:,ind)). But nor2ss(freq,ga,phib,cphi,ind)
%	gives a D matrix that corresponds to the static output of the
%	truncated modes.
%
%	Static correction modes rather than a D matrix can be introduced to
%	account for the low frequency influence of truncated modes, by
%	specifying the roll-off frequency of those modes FC (which should be
%	a decade above the last mode kept). This corrects both displacement
%	and velocity outputs.
%
%	To obtain a velocity output, specify displacement CPHID and velocity
%	CPHIV modal outputs and use NOR2SS(OM,GA,PHIB,[CPHID CPHIV])
%
%	See also NOR2XF, NOR2RES

%	Etienne Balmes  4/25/92, 04/12/96
%       Copyright (c) 1990-1996 by Etienne Balmes
%       All Rights Reserved.

[np,i1]=size(om);freq=[];
if     i1==1 freq=om;   om=diag(om.^2);
elseif i1~=np error('OM is not square');
end

[i2,i3]=size(ga);damp=[];
if ~isempty(freq)
   if i2==1 & i3==1
	if np==1 disp('WARNING: damping ratio input assumed'); end
	damp=ga*ones(np,1);  ga=diag(2*freq.*damp);
   elseif i3==1
	damp=ga;             ga=diag(2*freq.*damp);
   end	
else
   if i1~=np|i2~=np error('GA is not consistent with OM'); end
end


if nargin<5 		ind=[1:np];		end % keeps all modes
if isempty(ind) 	ind=[1:np];		end % keeps all modes
ind=sort(ind);

[i4,na]=size(phib); if i4~=np error('PHIB is not consistent with OM'); end
[ns,i7]=size(cphi);

if i7==2*i4 	cphii=cphi(:,1:(i7/2));
		cvi=cphi(:,(i7/2+1):i7); % disp. and vel. sensors
elseif i7==i4 	cphii=cphi(:,1:i7);cvi=zeros(ns,i4);
else 		error('The dimensions of cphi and phib are incompatible');
end

[NMode]=length(ind);	NCMode=0;

% ----------------------------------------------------------------------------
% static correction computation

if length(ind)~=np & ~isempty(freq) % freq specified and static correction

  cind=1:np;cind(ind)=ind*0;cind=cind(find(cind));
  finv=diag(freq(cind).^(-2));
  dd=cphii(:,cind)*finv*phib(cind,:);
  dv=cvi(:,cind)*finv*phib(cind,:);
  if nargin==6
    NCMode=na;
    if fc<10*max(freq(ind))
	   fc=10*max(freq(ind));
	   disp(sprintf('The cut-off frequency was changed to %7.3e',fc))
    end
    d=zeros(ns,na);
  else
    d=dd;NCMode=0;
  end

elseif length(ind)~=np % om specified and static correction

  disp('Static correction only available with modal frequencies FREQ');
  d = zeros(ns,na);NCMode=0;

elseif length(ind)==np % nostatic correction
  d = zeros(ns,na);NCMode=0;

end

NT=(NMode+NCMode);

% ----------------------------------------------------------------------------
% Construction of the state space matrix
% ----------------------------------------------------------------------------

   a=zeros(2*NT,2*NT);		a(1:NT,NT+[1:NT])=eye(NT,NT);
   b=zeros(2*NT,na);		c=zeros(ns,2*NT);

   a(NT+[1:NMode],[1:NMode])    = -om(ind,ind);
   a(NT+[1:NMode],NT+[1:NMode]) = -ga(ind,ind);

   b(NT+[1:NMode],:)	        = phib(ind,:);
   c(:,[1:NMode])		= cphii(:,ind);
   c(:,NT+[1:NMode])		= cvi(:,ind);

 if NCMode>0
   a(NT+NMode+[1:NCMode],NMode+[1:NCMode])   = -fc^2*eye(NCMode,NCMode);
   a(NT+NMode+[1:NCMode],NT+NMode+[1:NCMode])= -fc*eye(NCMode,NCMode);
   b(NT+NMode+[1:NCMode],:)		     = eye(NCMode,NCMode);
   c(:,NMode+[1:NCMode])		     = dd*fc^2;
   c(:,NT+NMode+[1:NCMode])		     = dv*fc^2;
 end






