 function [Ke,Me,Ce]=beam3d(ex,ey,ez,eo,ep)
% [Ke,Me]=beam3d(ex,ey,ez,eo,ep)
% [Ke,Me,Ce]=beam3d(ex,ey,ez,eo,ep)
%----------------------------------------------------------------
%    PURPOSE
%       Calculate the stiffness matrix Ke and the mass matrix Me
%       for a 3D elastic Bernoulli beam element. 
% 
%    INPUT:  ex = [x1 x2]        
%            ey = [y1 y2]   
%            ez = [z1 z2]       node coordinates  
%
%            eo = [xz yz zz];   orientation of local z axis
%
%            ep = [E G A Iy Iz Kv m (a b)];
%
%                               element properties 
%                            E: Young's modulus
%                            G: Shear modulus 
%                            A: Cross section area
%                           Iy: moment of inertia,local y-axis
%                           Iz: moment of inertia,local z-axis
%                           Kv: Saint-Venant's torsion constant
%                            m: rho*A mass per unit length
%                          a,b: damping coefficients,
%                               Ce=aMe+bKe 
% 
%
%    OUTPUT: Ke : beam stiffness matrix (12 x 12)
%            Me : element mass matrix 
%
%-----------------------------------------------------------------  

% LAST MODIFIED: I. Kjelberg of CSEM SA (CH)  1997-apr-30 
% Copyright (c)  Partly Division of Structural Mechanics and
%                Department of Solid Mechanics.
%                Lund Institute of Technology
%-------------------------------------------------------------


  b=[ ex(2)-ex(1); ey(2)-ey(1); ez(2)-ez(1) ];
  L=sqrt(b'*b);  n1=b/L;

  lc=sqrt(eo*eo'); n3=eo/lc;
  %
    E=ep(1); Gs=ep(2);
    A=ep(3);
    Iy=ep(4); Iz=ep(5);
    Kv=ep(6);
    m=ep(7);
    if length(ep)==9 
      aa=ep(8) ; bb=ep(9) ;
    else
      aa = 0; 	bb = 0;
    end
 
    a=E*A/L       ; b=12*E*Iz/L^3 ; c=6*E*Iz/L^2;
    d=12*E*Iy/L^3 ; e=6*E*Iy/L^2  ; f=Gs*Kv/L;
    g=2*E*Iy/L    ; h=2*E*Iz/L    ; k=(Iy+Iz)/A;

    Kle=[a  0  0  0  0  0 -a  0  0  0  0  0 ;
         0  b  0  0  0  c  0 -b  0  0  0  c ;
         0  0  d  0 -e  0  0  0 -d  0 -e  0 ;
         0  0  0  f  0  0  0  0  0 -f  0  0 ;
         0  0 -e  0 2*g 0  0  0  e  0  g  0 ;
         0  c  0  0  0 2*h 0 -c  0  0  0  h ;
        -a  0  0  0  0  0  a  0  0  0  0  0 ;
         0 -b  0  0  0 -c  0  b  0  0  0 -c ;
         0  0 -d  0  e  0  0  0  d  0  e  0 ;
         0  0  0 -f  0  0  0  0  0  f  0  0 ;
         0  0 -e  0  g  0  0  0  e  0 2*g 0 ;
         0  c  0  0  0  h  0 -c  0  0  0 2*h];
%
   Mle=m*L/420*[
        140   0    0    0    0    0   70    0    0    0    0    0   ;
         0   156   0    0    0   22*L  0   54    0    0    0  -13*L ;
         0    0   156   0  -22*L  0    0    0   54    0   13*L  0   ;
         0    0    0  140*k  0    0    0    0    0   70*k  0    0   ;
         0    0  -22*L  0   4*L^2 0    0    0 -13*L   0  -3*L^2 0   ;
         0   22*L  0    0    0  4*L^2  0  13*L   0    0    0  -3*L^2 ;
        70    0    0    0    0    0   140   0    0    0    0    0   ;
         0   54    0    0    0   13*L  0   156   0    0    0  -22*L ;
         0    0   54    0  -13*L  0    0    0   156   0   22*L  0   ;
         0    0    0  70*k   0    0    0    0    0  140*k  0    0   ;
         0    0  13*L   0 -3*L^2  0    0    0   22*L  0  4*L^2  0   ;
         0  -13*L  0    0    0 -3*L^2  0 -22*L   0    0    0  4*L^2 ];
 
%
    n2(1)=n3(2)*n1(3)-n3(3)*n1(2);
    n2(2)=-n1(3)*n3(1)+n1(1)*n3(3);
    n2(3)=n3(1)*n1(2)-n1(1)*n3(2);
%
    An=[n1';
        n2;
        n3];
%
%
   Cle=aa*Mle+bb*Kle;
%
    G=[  An     zeros(3) zeros(3) zeros(3);
       zeros(3)   An     zeros(3) zeros(3);
       zeros(3) zeros(3)   An     zeros(3);
       zeros(3) zeros(3) zeros(3)   An    ];
%
   Ke1=G'*Kle*G;    Me1=G'*Mle*G;    Ce1=G'*Cle*G;
 %----------------------------------------------------------   
    if nargout==0
      disp('Element stiffness matrix: ');
      disp(Ke1);
      disp('Element mass matrix: ');
      disp(Me1);
      return 
    end
    Ke=Ke1;
    Me=Me1;
    Ce=Ce1;
 %-------------------------- end -------------------------------
